import { useState } from "react";
import Header from "@/components/layout/Header";
import Footer from "@/components/layout/Footer";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Card } from "@/components/ui/card";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Checkbox } from "@/components/ui/checkbox";
import { Users, Award, TrendingUp, Heart, Upload } from "lucide-react";
import { useToast } from "@/hooks/use-toast";

const TrabalheConosco = () => {
  const [formData, setFormData] = useState({
    name: "",
    email: "",
    phone: "",
    role: "",
    message: "",
    acceptTerms: false,
  });
  const { toast } = useToast();

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!formData.acceptTerms) {
      toast({
        title: "Erro",
        description: "Por favor, aceite os termos da LGPD.",
        variant: "destructive",
      });
      return;
    }

    toast({
      title: "Candidatura enviada!",
      description: "Entraremos em contato em breve. Obrigado!",
    });

    // Reset form
    setFormData({
      name: "",
      email: "",
      phone: "",
      role: "",
      message: "",
      acceptTerms: false,
    });
  };

  const benefits = [
    {
      icon: Award,
      title: "Crescimento Profissional",
      description: "Treinamentos contínuos e certificações na área",
    },
    {
      icon: TrendingUp,
      title: "Plano de Carreira",
      description: "Oportunidades reais de crescimento interno",
    },
    {
      icon: Heart,
      title: "Bem-estar",
      description: "Ambiente de trabalho saudável e respeitoso",
    },
    {
      icon: Users,
      title: "Equipe Unida",
      description: "Trabalhe com profissionais experientes e dedicados",
    },
  ];

  const openPositions = [
    {
      title: "Técnico em Controle de Pragas",
      type: "CLT - Tempo Integral",
      requirements: [
        "Ensino médio completo",
        "CNH categoria B",
        "Disponibilidade para viagens regionais",
        "Experiência na área (desejável)",
      ],
    },
    {
      title: "Auxiliar de Limpeza de Reservatórios",
      type: "CLT - Tempo Integral",
      requirements: [
        "Ensino fundamental completo",
        "Não ter medo de altura",
        "Disponibilidade para trabalho aos sábados",
        "Experiência prévia (diferencial)",
      ],
    },
    {
      title: "Assistente Administrativo",
      type: "CLT - Tempo Integral",
      requirements: [
        "Ensino médio completo",
        "Conhecimento em pacote Office",
        "Experiência com atendimento ao cliente",
        "Organização e proatividade",
      ],
    },
  ];

  return (
    <div className="min-h-screen">
      <Header />
      <main>
        {/* Hero */}
        <section className="pt-20 pb-20 bg-gradient-to-br from-secondary to-secondary/80 text-white">
          <div className="container mx-auto px-4 text-center">
            <h1 className="mb-6">Trabalhe Conosco</h1>
            <p className="text-xl text-gray-200 max-w-3xl mx-auto">
              Faça parte de uma equipe comprometida com excelência, segurança e desenvolvimento
              profissional.
            </p>
          </div>
        </section>

        {/* Por que trabalhar conosco */}
        <section className="py-16 bg-background">
          <div className="container mx-auto px-4">
            <div className="text-center mb-12">
              <h2 className="mb-4">Por Que Trabalhar na Saúde Ambiental?</h2>
              <p className="text-lg text-muted-foreground max-w-2xl mx-auto">
                Oferecemos um ambiente de trabalho que valoriza o crescimento profissional e pessoal
                de cada colaborador.
              </p>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
              {benefits.map((benefit, index) => (
                <Card
                  key={benefit.title}
                  className="p-6 text-center hover:shadow-lg transition-all animate-fade-in-up"
                  style={{ animationDelay: `${index * 100}ms` }}
                >
                  <div className="inline-flex items-center justify-center w-14 h-14 bg-primary/10 rounded-full mb-4">
                    <benefit.icon className="w-7 h-7 text-primary" />
                  </div>
                  <h3 className="text-xl font-bold mb-2">{benefit.title}</h3>
                  <p className="text-sm text-muted-foreground">{benefit.description}</p>
                </Card>
              ))}
            </div>
          </div>
        </section>

        {/* Vagas Abertas */}
        <section className="py-16 bg-muted">
          <div className="container mx-auto px-4">
            <div className="text-center mb-12">
              <h2 className="mb-4">Vagas Abertas</h2>
              <p className="text-lg text-muted-foreground">
                Confira nossas oportunidades e candidate-se!
              </p>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-3 gap-8 mb-12">
              {openPositions.map((position, index) => (
                <Card
                  key={position.title}
                  className="p-6 hover:shadow-xl transition-all animate-fade-in-up"
                  style={{ animationDelay: `${index * 100}ms` }}
                >
                  <h3 className="text-xl font-bold mb-2">{position.title}</h3>
                  <p className="text-sm text-primary mb-4">{position.type}</p>
                  <div>
                    <p className="text-sm font-semibold mb-2">Requisitos:</p>
                    <ul className="text-sm text-muted-foreground space-y-1">
                      {position.requirements.map((req, i) => (
                        <li key={i}>• {req}</li>
                      ))}
                    </ul>
                  </div>
                </Card>
              ))}
            </div>
          </div>
        </section>

        {/* Formulário de Candidatura */}
        <section className="py-16 bg-background">
          <div className="container mx-auto px-4">
            <div className="max-w-2xl mx-auto">
              <div className="text-center mb-12">
                <h2 className="mb-4">Candidate-se</h2>
                <p className="text-muted-foreground">
                  Preencha o formulário abaixo e anexe seu currículo. Entraremos em contato em breve!
                </p>
              </div>

              <Card className="p-8">
                <form onSubmit={handleSubmit} className="space-y-6">
                  <div>
                    <Label htmlFor="name">Nome Completo *</Label>
                    <Input
                      id="name"
                      type="text"
                      value={formData.name}
                      onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                      required
                    />
                  </div>

                  <div>
                    <Label htmlFor="email">E-mail *</Label>
                    <Input
                      id="email"
                      type="email"
                      value={formData.email}
                      onChange={(e) => setFormData({ ...formData, email: e.target.value })}
                      required
                    />
                  </div>

                  <div>
                    <Label htmlFor="phone">Telefone/WhatsApp *</Label>
                    <Input
                      id="phone"
                      type="tel"
                      placeholder="(81) 99999-9999"
                      value={formData.phone}
                      onChange={(e) => setFormData({ ...formData, phone: e.target.value })}
                      required
                    />
                  </div>

                  <div>
                    <Label htmlFor="role">Vaga de Interesse *</Label>
                    <Select
                      value={formData.role}
                      onValueChange={(value) => setFormData({ ...formData, role: value })}
                      required
                    >
                      <SelectTrigger>
                        <SelectValue placeholder="Selecione uma vaga" />
                      </SelectTrigger>
                      <SelectContent>
                        {openPositions.map((position) => (
                          <SelectItem key={position.title} value={position.title}>
                            {position.title}
                          </SelectItem>
                        ))}
                        <SelectItem value="outro">Outra oportunidade</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>

                  <div>
                    <Label htmlFor="cv">Currículo (PDF) *</Label>
                    <div className="mt-2 flex items-center justify-center border-2 border-dashed border-border rounded-lg p-6 hover:border-primary transition-colors cursor-pointer">
                      <div className="text-center">
                        <Upload className="w-10 h-10 text-muted-foreground mx-auto mb-2" />
                        <p className="text-sm text-muted-foreground">
                          Clique para fazer upload ou arraste seu currículo aqui
                        </p>
                        <p className="text-xs text-muted-foreground mt-1">PDF, máximo 5MB</p>
                      </div>
                    </div>
                  </div>

                  <div>
                    <Label htmlFor="message">Mensagem (Opcional)</Label>
                    <Textarea
                      id="message"
                      rows={4}
                      placeholder="Conte-nos um pouco sobre você e sua experiência..."
                      value={formData.message}
                      onChange={(e) => setFormData({ ...formData, message: e.target.value })}
                    />
                  </div>

                  <div className="flex items-start space-x-2">
                    <Checkbox
                      id="terms"
                      checked={formData.acceptTerms}
                      onCheckedChange={(checked) =>
                        setFormData({ ...formData, acceptTerms: checked as boolean })
                      }
                    />
                    <label htmlFor="terms" className="text-sm text-muted-foreground leading-tight">
                      Autorizo o uso dos meus dados pessoais conforme a Lei Geral de Proteção de
                      Dados (LGPD) para fins de recrutamento e seleção. *
                    </label>
                  </div>

                  <Button type="submit" variant="hero" size="lg" className="w-full">
                    Enviar Candidatura
                  </Button>
                </form>
              </Card>
            </div>
          </div>
        </section>
      </main>
      <Footer />
    </div>
  );
};

export default TrabalheConosco;
