import { useState } from "react";
import Header from "@/components/layout/Header";
import Footer from "@/components/layout/Footer";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Card } from "@/components/ui/card";
import { Checkbox } from "@/components/ui/checkbox";
import { MapPin, Phone, Mail, Clock, MessageCircle } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { company } from "@/config/company";

const Contato = () => {
  const [formData, setFormData] = useState({
    name: "",
    email: "",
    phone: "",
    message: "",
    acceptTerms: false,
  });
  const { toast } = useToast();

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();

    if (!formData.acceptTerms) {
      toast({
        title: "Erro",
        description: "Por favor, aceite os termos da LGPD.",
        variant: "destructive",
      });
      return;
    }

    toast({
      title: "Mensagem enviada!",
      description: "Entraremos em contato em breve. Obrigado!",
    });

    // Reset form
    setFormData({
      name: "",
      email: "",
      phone: "",
      message: "",
      acceptTerms: false,
    });
  };

  const whatsappNumber = company.whatsapps.vendas.replace(/\D/g, "");
  const whatsappLink = `https://wa.me/${whatsappNumber}?text=Olá!%20Gostaria%20de%20mais%20informações.`;

  const contactInfo = [
    {
      icon: MapPin,
      title: "Endereço",
      details: [
        "Rua Coronel Mizael de Mendonça, 94 - San Martin",
        "Recife, PE - CEP 50761-190",
      ],
    },
    {
      icon: Phone,
      title: "Telefone",
      details: ["(81) 3023-0800"],
      links: ["tel:+558130230800"],
    },
    {
      icon: MessageCircle,
      title: "WhatsApp",
      details: ["(81) 3023-0800", "(81) 99857-8596 (Vendas)", "(81) 98263-6946 (Operacional)"],
      links: [
        "https://wa.me/558130230800",
        "https://wa.me/5581998578596",
        "https://wa.me/5581982636946",
      ],
    },
    {
      icon: Mail,
      title: "E-mail",
      details: ["vendas@hecol.com.br", "qualidade@hecol.com.br", "financeiro@hecol.com.br"],
      links: [
        "mailto:vendas@hecol.com.br",
        "mailto:qualidade@hecol.com.br",
        "mailto:financeiro@hecol.com.br",
      ],
    },
    {
      icon: Clock,
      title: "Horário de Atendimento",
      details: ["Serviços 24 horas"],
    },
  ];

  return (
    <div className="min-h-screen">
      <Header />
      <main>
        {/* Hero */}
        <section className="pt-24 pb-20 bg-gradient-to-br from-secondary to-secondary/80 text-white">
          <div className="container mx-auto px-4 text-center">
            <h1 className="mb-6">Entre em Contato</h1>
            <p className="text-xl text-gray-200 max-w-3xl mx-auto">
              Estamos prontos para atendê-lo. Fale conosco por telefone, WhatsApp ou pelo formulário
              abaixo.
            </p>
          </div>
        </section>

        {/* Informações de Contato */}
        <section className="py-16 bg-background">
          <div className="container mx-auto px-4">
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-16">
              {contactInfo.map((info, index) => (
                <Card
                  key={info.title}
                  className="p-6 text-center hover:shadow-lg transition-all animate-fade-in-up"
                  style={{ animationDelay: `${index * 100}ms` }}
                >
                  <div className="inline-flex items-center justify-center w-14 h-14 bg-primary/10 rounded-full mb-4">
                    <info.icon className="w-7 h-7 text-primary" />
                  </div>
                  <h3 className="font-bold mb-3">{info.title}</h3>
                  {info.links
                    ? info.details.map((detail, i) => (
                        <a
                          key={i}
                          href={info.links?.[i]}
                          className="block text-sm text-muted-foreground hover:text-primary transition-colors"
                        >
                          {detail}
                        </a>
                      ))
                    : info.details.map((detail, i) => (
                        <p key={i} className="text-sm text-muted-foreground">
                          {detail}
                        </p>
                      ))}
                </Card>
              ))}
            </div>

            {/* WhatsApp CTA */}
            <div className="text-center mb-16">
              <Card className="inline-block p-6 bg-[#25D366]/10 border-[#25D366]/20">
                <MessageCircle className="w-12 h-12 text-[#25D366] mx-auto mb-3" />
                <h3 className="text-xl font-bold mb-2">Atendimento Rápido via WhatsApp</h3>
                <p className="text-muted-foreground mb-4">
                  Fale diretamente com nossa equipe e receba resposta imediata!
                </p>
                <Button variant="whatsapp" size="lg" asChild>
                  <a href={whatsappLink} target="_blank" rel="noopener noreferrer">
                    <Phone className="w-5 h-5" />
                    Abrir WhatsApp
                  </a>
                </Button>
              </Card>
            </div>
          </div>
        </section>

        {/* Formulário e Mapa */}
        <section className="py-16 bg-muted">
          <div className="container mx-auto px-4">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-12">
              {/* Formulário */}
              <div>
                <h2 className="mb-6">Envie uma Mensagem</h2>
                <Card className="p-8">
                  <form onSubmit={handleSubmit} className="space-y-4">
                    <div>
                      <Label htmlFor="name">Nome Completo *</Label>
                      <Input
                        id="name"
                        type="text"
                        value={formData.name}
                        onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                        required
                      />
                    </div>

                    <div>
                      <Label htmlFor="email">E-mail *</Label>
                      <Input
                        id="email"
                        type="email"
                        value={formData.email}
                        onChange={(e) => setFormData({ ...formData, email: e.target.value })}
                        required
                      />
                    </div>

                    <div>
                      <Label htmlFor="phone">Telefone *</Label>
                      <Input
                        id="phone"
                        type="tel"
                        placeholder="(81) 99999-9999"
                        value={formData.phone}
                        onChange={(e) => setFormData({ ...formData, phone: e.target.value })}
                        required
                      />
                    </div>

                    <div>
                      <Label htmlFor="message">Mensagem *</Label>
                      <Textarea
                        id="message"
                        rows={5}
                        placeholder="Como podemos ajudá-lo?"
                        value={formData.message}
                        onChange={(e) => setFormData({ ...formData, message: e.target.value })}
                        required
                      />
                    </div>

                    <div className="flex items-start space-x-2">
                      <Checkbox
                        id="terms"
                        checked={formData.acceptTerms}
                        onCheckedChange={(checked) =>
                          setFormData({ ...formData, acceptTerms: checked as boolean })
                        }
                      />
                      <label
                        htmlFor="terms"
                        className="text-sm text-muted-foreground leading-tight"
                      >
                        Autorizo o uso dos meus dados pessoais conforme a Lei Geral de Proteção de
                        Dados (LGPD) para fins de contato. *
                      </label>
                    </div>

                    <Button type="submit" variant="hero" size="lg" className="w-full">
                      Enviar Mensagem
                    </Button>
                  </form>
                </Card>
              </div>

              {/* Mapa */}
              <div>
                <h2 className="mb-6">Nossa Localização</h2>
                <Card className="overflow-hidden h-full min-h-[500px]">
                  <div className="w-full h-full bg-muted flex items-center justify-center">
                    <div className="text-center p-8">
                      <MapPin className="w-16 h-16 text-primary mx-auto mb-4" />
                      <p className="text-muted-foreground">
                        Mapa interativo da localização seria exibido aqui
                      </p>
                      <p className="text-sm text-muted-foreground mt-2">
                        Rua Coronel Mizael de Mendonça, 94 - San Martin
                        <br />
                        Recife, PE - CEP 50761-190
                      </p>
                    </div>
                  </div>
                </Card>
              </div>
            </div>
          </div>
        </section>

        {/* Dúvidas Frequentes Rápidas */}
        <section className="py-16 bg-background">
          <div className="container mx-auto px-4 text-center">
            <h2 className="mb-12">Dúvidas Frequentes</h2>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
              <Card className="p-6">
                <h3 className="font-bold mb-2">Atendimento 24h?</h3>
                <p className="text-sm text-muted-foreground">
                  Sim! Serviços 24 horas para emergências.
                </p>
              </Card>
              <Card className="p-6">
                <h3 className="font-bold mb-2">Orçamento em até 24h?</h3>
                <p className="text-sm text-muted-foreground">
                  Sim. Solicite pelo WhatsApp de vendas e receba rápido.
                </p>
              </Card>
              <Card className="p-6">
                <h3 className="font-bold mb-2">Cobertura geográfica</h3>
                <p className="text-sm text-muted-foreground">
                  Cobrimos toda a região Nordeste.
                </p>
              </Card>
            </div>
          </div>
        </section>
      </main>
      <Footer />
    </div>
  );
};

export default Contato;
