import { useState, useEffect, useCallback } from 'react';

export interface CookieConsentPreferences {
  necessary: boolean;
  preferences: boolean;
  analytics: boolean;
  marketing: boolean;
  version: string;
  timestamp: string;
  locale: string;
}

const COOKIE_NAME = 'hecol_consent';
const CONSENT_VERSION = '1.0';
const CONSENT_DURATION_DAYS = 365;

// Detecta Do Not Track ou Global Privacy Control
const detectDoNotTrack = (): boolean => {
  return (
    navigator.doNotTrack === '1' ||
    (window as any).doNotTrack === '1' ||
    (navigator as any).msDoNotTrack === '1' ||
    (navigator as any).globalPrivacyControl === true
  );
};

// Google Consent Mode v2
declare global {
  interface Window {
    gtag?: (...args: any[]) => void;
    dataLayer?: any[];
  }
}

const updateGoogleConsent = (preferences: CookieConsentPreferences) => {
  if (typeof window.gtag === 'function') {
    window.gtag('consent', 'update', {
      ad_storage: preferences.marketing ? 'granted' : 'denied',
      analytics_storage: preferences.analytics ? 'granted' : 'denied',
      personalization_storage: preferences.preferences ? 'granted' : 'denied',
      security_storage: 'granted', // Always granted for security
    });
  }
};

export const useCookieConsent = () => {
  const [consent, setConsent] = useState<CookieConsentPreferences | null>(null);
  const [showBanner, setShowBanner] = useState(false);

  useEffect(() => {
    // Inicializa Google Consent Mode v2 com denied (opt-in)
    if (typeof window.gtag === 'function') {
      window.gtag('consent', 'default', {
        ad_storage: 'denied',
        analytics_storage: 'denied',
        personalization_storage: 'denied',
        security_storage: 'granted',
        wait_for_update: 500,
      });
    }

    const savedConsent = localStorage.getItem(COOKIE_NAME);
    
    if (savedConsent) {
      try {
        const parsed: CookieConsentPreferences = JSON.parse(savedConsent);
        
        // Verifica se precisa re-consent (mudança de versão)
        if (parsed.version !== CONSENT_VERSION) {
          setShowBanner(true);
          return;
        }

        setConsent(parsed);
        updateGoogleConsent(parsed);
        loadScripts(parsed);
      } catch (error) {
        console.error('Error parsing consent:', error);
        setShowBanner(true);
      }
    } else {
      // Verifica Do Not Track
      if (detectDoNotTrack()) {
        const dntPreferences: CookieConsentPreferences = {
          necessary: true,
          preferences: false,
          analytics: false,
          marketing: false,
          version: CONSENT_VERSION,
          timestamp: new Date().toISOString(),
          locale: navigator.language || 'pt-BR',
        };
        saveConsent(dntPreferences);
        setConsent(dntPreferences);
      } else {
        setShowBanner(true);
      }
    }
  }, []);

  const loadScripts = (preferences: CookieConsentPreferences) => {
    // Carrega scripts de acordo com o consentimento
    if (preferences.analytics) {
      // Carregar Google Analytics / GA4
      // TODO: Adicionar seu GA4 ID aqui
      // loadGoogleAnalytics('G-XXXXXXXXXX');
    }

    if (preferences.marketing) {
      // Carregar Meta Pixel, Google Ads, etc.
      // TODO: Adicionar seus pixels de marketing aqui
    }

    // Preferências podem carregar scripts de idioma/tema
    if (preferences.preferences) {
      // Scripts de personalização
    }
  };

  const saveConsent = useCallback((preferences: CookieConsentPreferences) => {
    try {
      localStorage.setItem(COOKIE_NAME, JSON.stringify(preferences));
      
      // Define cookie com segurança
      const expiryDate = new Date();
      expiryDate.setDate(expiryDate.getDate() + CONSENT_DURATION_DAYS);
      
      document.cookie = `${COOKIE_NAME}=1; expires=${expiryDate.toUTCString()}; path=/; SameSite=Strict; Secure`;
      
      setConsent(preferences);
      updateGoogleConsent(preferences);
      loadScripts(preferences);
    } catch (error) {
      console.error('Error saving consent:', error);
    }
  }, []);

  const acceptAll = useCallback(() => {
    const preferences: CookieConsentPreferences = {
      necessary: true,
      preferences: true,
      analytics: true,
      marketing: true,
      version: CONSENT_VERSION,
      timestamp: new Date().toISOString(),
      locale: navigator.language || 'pt-BR',
    };
    saveConsent(preferences);
    setShowBanner(false);
  }, [saveConsent]);

  const rejectAll = useCallback(() => {
    const preferences: CookieConsentPreferences = {
      necessary: true,
      preferences: false,
      analytics: false,
      marketing: false,
      version: CONSENT_VERSION,
      timestamp: new Date().toISOString(),
      locale: navigator.language || 'pt-BR',
    };
    saveConsent(preferences);
    setShowBanner(false);
  }, [saveConsent]);

  const savePreferences = useCallback((prefs: Partial<CookieConsentPreferences>) => {
    const preferences: CookieConsentPreferences = {
      necessary: true, // Always true
      preferences: prefs.preferences ?? false,
      analytics: prefs.analytics ?? false,
      marketing: prefs.marketing ?? false,
      version: CONSENT_VERSION,
      timestamp: new Date().toISOString(),
      locale: navigator.language || 'pt-BR',
    };
    saveConsent(preferences);
    setShowBanner(false);
  }, [saveConsent]);

  const openPreferences = useCallback(() => {
    setShowBanner(true);
  }, []);

  return {
    consent,
    showBanner,
    setShowBanner,
    acceptAll,
    rejectAll,
    savePreferences,
    openPreferences,
  };
};
