import { useState, useEffect } from "react";
import { X, ChevronDown } from "lucide-react";
import { useLanguage } from "@/lib/i18n/LanguageContext";

const PromoBar = () => {
  const [isVisible, setIsVisible] = useState(true);
  const [isExpanded, setIsExpanded] = useState(false);
  const { t } = useLanguage();

  useEffect(() => {
    let timeout: NodeJS.Timeout;
    if (isExpanded) {
      timeout = setTimeout(() => setIsExpanded(false), 3000);
    }
    return () => clearTimeout(timeout);
  }, [isExpanded]);

  if (!isVisible) return null;

  return (
    <div 
      className={`bg-promo text-white transition-all duration-200 ${
        isExpanded ? 'h-20 sm:h-[90px]' : 'h-7 sm:h-8'
      }`}
      onMouseEnter={() => setIsExpanded(true)}
      onMouseLeave={() => setIsExpanded(false)}
      onFocus={() => setIsExpanded(true)}
      onBlur={(e) => {
        if (!e.currentTarget.contains(e.relatedTarget)) {
          setIsExpanded(false);
        }
      }}
    >
      <div className="container mx-auto px-4 h-full flex items-center justify-between relative">
        {/* Collapsed View */}
        <div className={`flex items-center justify-center w-full transition-opacity duration-200 ${
          isExpanded ? 'opacity-0 absolute' : 'opacity-100'
        }`}>
          <p className="text-xs sm:text-sm font-medium">
            {t.promo.text}
          </p>
          <button
            className="sm:hidden ml-2"
            onClick={() => setIsExpanded(!isExpanded)}
            aria-label="Expandir promoção"
          >
            <ChevronDown className={`w-4 h-4 transition-transform ${isExpanded ? 'rotate-180' : ''}`} />
          </button>
        </div>

        {/* Expanded View */}
        <div className={`w-full transition-opacity duration-200 ${
          isExpanded ? 'opacity-100' : 'opacity-0 absolute pointer-events-none'
        }`}>
          <div className="flex flex-col sm:flex-row items-center justify-center gap-2 sm:gap-4">
            <p className="text-sm sm:text-base font-semibold">
              {t.promo.text}
            </p>
            <a
              href="/contato"
              className="text-xs sm:text-sm bg-white text-promo px-4 py-1.5 rounded-md font-medium hover:bg-brand-red hover:text-white hover:scale-105 transition-all duration-200 focus-visible:ring-2 focus-visible:ring-brand-red-focus focus-visible:ring-offset-2 focus-visible:ring-offset-promo motion-reduce:hover:scale-100"
            >
              Ver Ofertas
            </a>
          </div>
        </div>

        {/* Close Button */}
        <button
          onClick={() => setIsVisible(false)}
          className="absolute right-4 p-1 hover:bg-white/20 rounded transition-colors z-10"
          aria-label="Fechar"
        >
          <X className="w-3 h-3 sm:w-4 sm:h-4" />
        </button>
      </div>
    </div>
  );
};

export default PromoBar;
