import { useEffect, useState, useRef } from "react";
import { Users, Clock, Award, Building } from "lucide-react";

const stats = [
  {
    icon: Users,
    value: 5000,
    suffix: "+",
    label: "Clientes Atendidos",
  },
  {
    icon: Clock,
    value: 2,
    suffix: "h",
    label: "Tempo Médio de Resposta",
  },
  {
    icon: Award,
    value: 15,
    suffix: "+",
    label: "Anos de Experiência",
  },
  {
    icon: Building,
    value: 98,
    suffix: "%",
    label: "Satisfação dos Clientes",
  },
];

const Counter = ({ end, suffix, duration = 2000 }: { end: number; suffix: string; duration?: number }) => {
  const [count, setCount] = useState(0);
  const [isVisible, setIsVisible] = useState(false);
  const counterRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting && !isVisible) {
          setIsVisible(true);
        }
      },
      { threshold: 0.5 }
    );

    if (counterRef.current) {
      observer.observe(counterRef.current);
    }

    return () => observer.disconnect();
  }, [isVisible]);

  useEffect(() => {
    if (!isVisible) return;

    let startTime: number | null = null;
    const step = (timestamp: number) => {
      if (!startTime) startTime = timestamp;
      const progress = Math.min((timestamp - startTime) / duration, 1);
      setCount(Math.floor(progress * end));
      if (progress < 1) {
        requestAnimationFrame(step);
      }
    };
    requestAnimationFrame(step);
  }, [isVisible, end, duration]);

  return (
    <div ref={counterRef} className="text-4xl md:text-5xl font-bold text-primary">
      {count}
      {suffix}
    </div>
  );
};

const Numbers = () => {
  return (
    <section className="bg-background border-t-2 border-brand-red-divider">
      <div className="container mx-auto px-4 py-20">
        <div className="text-center mb-16">
          <h2 className="mb-4">Nossos Números</h2>
          <p className="text-lg text-muted-foreground max-w-2xl mx-auto">
            Resultados que comprovam nossa excelência e compromisso com a qualidade.
          </p>
        </div>

        <div className="grid grid-cols-2 lg:grid-cols-4 gap-8">
          {stats.map((stat, index) => (
            <div
              key={stat.label}
              className="text-center p-6 rounded-lg hover:bg-muted transition-colors animate-fade-in-up"
              style={{ animationDelay: `${index * 100}ms` }}
            >
              <div className="inline-flex items-center justify-center w-16 h-16 bg-primary/10 rounded-full mb-4">
                <stat.icon className="w-8 h-8 text-primary" />
              </div>
              <Counter end={stat.value} suffix={stat.suffix} />
              <p className="text-sm text-muted-foreground mt-2 font-medium">{stat.label}</p>
            </div>
          ))}
        </div>
      </div>
    </section>
  );
};

export default Numbers;
