import { useLanguage } from "@/lib/i18n/LanguageContext";
import { useEffect, useState, useMemo } from "react";
import { loadManifest, getAssetsBySection, getAssetPath, type ManifestRow } from "@/lib/contentLoader";

const ClientsTicker = () => {
  const { t } = useLanguage();
  const [clients, setClients] = useState<ManifestRow[]>([]);

  useEffect(() => {
    loadManifest().then(manifest => {
      const clientData = getAssetsBySection(manifest, 'clients');
      setClients(clientData);
    });
  }, []);

  // Shuffle arrays differently for each row
  const shuffledRow1 = useMemo(() => {
    if (clients.length === 0) return [];
    return [...clients].sort(() => Math.random() - 0.5);
  }, [clients]);

  const shuffledRow2 = useMemo(() => {
    if (clients.length === 0) return [];
    return [...clients].sort(() => Math.random() - 0.5);
  }, [clients]);

  if (clients.length === 0) return null;

  return (
    <section className="bg-muted overflow-hidden border-t-2 border-brand-red-divider pt-10 sm:pt-12" aria-label={t.clients.title}>
      <div className="container mx-auto px-4 pb-12 mb-4">
        <h2 className="text-3xl md:text-4xl font-bold text-center text-foreground mb-4">
          {t.clients.title}
        </h2>
        <p className="text-center text-muted-foreground max-w-2xl mx-auto">
          {t.clients.subtitle}
        </p>
      </div>

      {/* First row - Left to Right (faster) */}
      <div className="relative mb-12">
        <div className="flex animate-marquee hover:pause group" style={{ animationDuration: '25s' }}>
          {[...shuffledRow1, ...shuffledRow1].map((client, index) => (
            <div
              key={`row1-${index}`}
              className="flex-shrink-0 transition-transform duration-200 will-change-transform motion-reduce:transition-none hover:scale-110"
              style={{ marginLeft: index === 0 ? '0' : `${32 + (index % 3) * 8}px` }}
            >
              <img
                src={getAssetPath('clients', client.filename)}
                alt={client.alt_pt}
                className="h-10 sm:h-12 md:h-14 w-auto object-contain opacity-85 hover:opacity-100 transition-opacity duration-200"
                loading="lazy"
                onError={(e) => {
                  e.currentTarget.src = '/logo/logo-1-branca.png';
                }}
              />
            </div>
          ))}
        </div>
      </div>

      {/* Second row - Right to Left (slower) with offset */}
      <div className="relative mb-8">
        <div className="flex animate-marquee-reverse hover:pause group" style={{ animationDuration: '35s', animationDelay: '-5s' }}>
          {[...shuffledRow2, ...shuffledRow2].map((client, index) => (
            <div
              key={`row2-${index}`}
              className="flex-shrink-0 transition-transform duration-200 will-change-transform motion-reduce:transition-none hover:scale-110"
              style={{ marginLeft: index === 0 ? '0' : `${40 + (index % 4) * 6}px` }}
            >
              <img
                src={getAssetPath('clients', client.filename)}
                alt={client.alt_pt}
                className="h-10 sm:h-12 md:h-14 w-auto object-contain opacity-85 hover:opacity-100 transition-opacity duration-200"
                loading="lazy"
                onError={(e) => {
                  e.currentTarget.src = '/logo/logo-1-branca.png';
                }}
              />
            </div>
          ))}
        </div>
      </div>
    </section>
  );
};

export default ClientsTicker;
