import { useEffect, useState } from "react";
import { Card } from "@/components/ui/card";
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from "@/components/ui/tooltip";
import { loadManifest, getAssetsBySection, getAssetPath, type ManifestRow } from "@/lib/contentLoader";
import CertificationModal, { certificationData, type CertificationInfo } from "./CertificationModal";

const Certifications = () => {
  const [certifications, setCertifications] = useState<ManifestRow[]>([]);
  const [selectedCert, setSelectedCert] = useState<CertificationInfo | null>(null);
  const [modalOpen, setModalOpen] = useState(false);

  useEffect(() => {
    loadManifest().then(manifest => {
      const certData = getAssetsBySection(manifest, 'certifications');
      // Lista de certificações permitidas
      const allowedCerts = ['adagro', 'ibama', 'visa', 'vigilancia', 'dircon', 'seman', 'cprh', 'anvisa', 'crq', 'nr-31', 'nr-33', 'nr-35'];
      
      // Filtra apenas as certificações permitidas
      const filteredCerts = certData.filter(cert => {
        const name = cert.filename.toLowerCase();
        return allowedCerts.some(allowed => name.includes(allowed));
      });
      setCertifications(filteredCerts);
    });
  }, []);

  const getCertKey = (filename: string): string => {
    const name = filename.toLowerCase().replace(/\.(png|jpg|jpeg|gif|webp|avif)$/i, '');
    
    if (name.includes('adagro')) return 'adagro';
    if (name.includes('ibama')) return 'ibama';
    if (name.includes('visa') || name.includes('vigilancia')) return 'vigilancia';
    if (name.includes('dircon')) return 'dircon';
    if (name.includes('seman')) return 'seman';
    if (name.includes('cprh')) return 'cprh';
    if (name.includes('anvisa')) return 'anvisa';
    if (name.includes('crq')) return 'crq';
    if (name.includes('nr-31')) return 'nr-31';
    if (name.includes('nr-33')) return 'nr-33';
    if (name.includes('nr-35')) return 'nr-35';
    
    return 'vigilancia';
  };

  const handleCertClick = (cert: ManifestRow) => {
    const certKey = getCertKey(cert.filename);
    const certInfo = certificationData[certKey];
    if (certInfo) {
      setSelectedCert(certInfo);
      setModalOpen(true);
    }
  };

  if (certifications.length === 0) return null;
  
  return (
    <>
      <TooltipProvider>
        <section className="bg-muted border-t-2 border-brand-red-divider" id="certificacoes" aria-label="Certificações e Conformidades">
          <div className="container mx-auto px-4 py-20">
            <div className="text-center mb-16">
              <h2 className="mb-4">Certificações e Conformidades</h2>
              <p className="text-lg text-muted-foreground max-w-2xl mx-auto">
                Operamos em total conformidade com todas as exigências legais e técnicas, garantindo
                segurança e qualidade em cada serviço.
              </p>
            </div>

            {/* Certificações */}
            <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 xl:grid-cols-5 gap-6">
              {certifications.map((cert, index) => {
                const certKey = getCertKey(cert.filename);
                const certInfo = certificationData[certKey];
                
                return (
                  <Tooltip key={`${cert.filename}-${index}`}>
                    <TooltipTrigger asChild>
                      <Card
                        className="p-6 hover:shadow-lg transition-all duration-300 hover:-translate-y-1 animate-fade-in-up cursor-pointer hover:border-primary/50"
                        style={{ animationDelay: `${index * 50}ms` }}
                        onClick={() => handleCertClick(cert)}
                      >
                        <div className="flex flex-col items-center text-center space-y-4">
                          <div className="w-20 h-20 flex items-center justify-center">
                            <img 
                              src={getAssetPath('certifications', cert.filename)}
                              alt={cert.alt_pt}
                              className="w-full h-full object-contain"
                              onError={(e) => {
                                e.currentTarget.src = '/logo/logo-1-branca.png';
                              }}
                            />
                          </div>
                          <h3 className="font-semibold text-base">{cert.title_pt}</h3>
                        </div>
                      </Card>
                    </TooltipTrigger>
                    <TooltipContent side="top" className="max-w-xs">
                      <p className="text-sm">{certInfo?.what.substring(0, 100)}...</p>
                      <p className="text-xs text-muted-foreground mt-1">Clique para saber mais</p>
                    </TooltipContent>
                  </Tooltip>
                );
              })}
            </div>
          </div>
        </section>
      </TooltipProvider>

      <CertificationModal 
        open={modalOpen}
        onOpenChange={setModalOpen}
        certification={selectedCert}
      />
    </>
  );
};

export default Certifications;
